<?php

namespace App\Http\Controllers;

use App\Models\Location;
use App\Models\Stock;
use Yajra\DataTables\DataTables;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use App\Services\DataScopeService;

class StockController extends Controller
{

    public function index()
    {
        $totalItems = Stock::sum('available_qty');
        $inStock = Stock::where('available_qty', '>', 0)->count();
        $outOfStock = Stock::where('available_qty', 0)->count();
        $lowStock = Stock::whereColumn('available_qty', '<=', 'reorder_level')->count();

        return view('admin.stock.index', compact(
            'totalItems',
            'inStock',
            'outOfStock',
            'lowStock'
        ));
    }

//    public function getStock(Request $request)
// {
//     $user = auth()->user();

//     if (!$user) {
//         return response()->json(['status' => false, 'message' => 'Unauthorized'], 401);
//     }

//     $roleScopes = $user->role?->dataScopes
//         ->pluck('scope_type', 'module')
//         ->map(fn($s) => strtolower($s))
//         ->toArray() ?? [];

//     $locations = Location::withCount([
//         'stocks as total_items',
//         'stocks as in_stock' => function ($q) {
//             $q->where('available_qty', '>', 0);
//         },
//         'stocks as out_of_stock' => function ($q) {
//             $q->where('available_qty', '<=', 0);
//         }
//     ])->withSum('stocks as total_stock_qty', 'available_qty');

//     // ---------------------------
//     // Filter by user data scope
//     // ---------------------------
//     if (in_array('assigned_location', $roleScopes)) {
//         $locationIds = $user->locations()->pluck('location_id')->toArray();
//         $locations->whereIn('id', $locationIds);
//     }

//     if (in_array('assigned_franchise', $roleScopes)) {
//         $franchiseIds = $user->franchises()->pluck('id')->toArray();
//         $locations->whereIn('id', $franchiseIds);
//     }

//     if (in_array('assigned_dealer', $roleScopes)) {
//         $dealerIds = $user->dealers()->pluck('id')->toArray();
//         $locations->whereIn('id', $dealerIds);
//     }

//     if ($request->type) {
//         $locations->where('type', $request->type);
//     }

//     return DataTables::of($locations)
//         ->addIndexColumn()
//         ->addColumn('location', fn($row) => '<strong>' . $row->name . '</strong>')
//         ->addColumn('type', fn($row) => '<span class="text-muted">' . $row->type . '</span>')
//         ->addColumn('total_items', fn($row) => number_format($row->total_items))
//         ->addColumn('in_stock', fn($row) => '<span class="badge bg-success-subtle text-success">Available</span>')
//         ->addColumn('out_of_stock', fn($row) => '<span class="badge bg-danger-subtle text-danger">' . $row->out_of_stock . '</span>')
//         ->addColumn('total_stock_qty', fn($row) => number_format($row->total_stock_qty ?? 0))
//         ->addColumn('action', fn($row) => '
//             <a href="' . route('stock.view', $row->id) . '" class="btn btn-sm btn-light">
//                 <i class="ti ti-eye"></i>
//             </a>
//             <a href="' . route('stock.edit', $row->id) . '" class="btn btn-sm btn-warning ms-1">
//                 <i class="ti ti-edit"></i>
//             </a>
//         ')
//         ->rawColumns(['type', 'location', 'in_stock', 'out_of_stock', 'action'])
//         ->make(true);
// }

  public function getStock(Request $request)
{
    $user = auth()->user();

    if (!$user) {
        return response()->json(['status' => false, 'message' => 'Unauthorized'], 401);
    }

    $roleScopes = $user->role?->dataScopes
        ->pluck('scope_type', 'module')
        ->map(fn($s) => strtolower($s))
        ->toArray() ?? [];

    $locations = Location::withCount([
        'stocks as total_items',
        'stocks as in_stock' => function ($q) {
            $q->where('available_qty', '>', 0);
        },
        'stocks as out_of_stock' => function ($q) {
            $q->where('available_qty', '<=', 0);
        }
    ])->withSum('stocks as total_stock_qty', 'available_qty');

    // ---------------------------
    // Filter by user data scope
    // ---------------------------
   $scopeTypes = array_values($roleScopes);

if (in_array('assigned_location', $scopeTypes)) {
    $locationIds = $user->locations()->pluck('locations.id');
    $locations->whereIn('id', $locationIds);
}

if (in_array('assigned_franchise', $scopeTypes)) {
    $franchiseIds = $user->franchises()->pluck('locations.id');
    $locations->whereIn('id', $franchiseIds);
}

if (in_array('assigned_dealer', $scopeTypes)) {
    $dealerIds = $user->dealers()->pluck('locations.id');
    $locations->whereIn('id', $dealerIds);
}


    if ($request->type) {
        $locations->where('type', $request->type);
    }

    return DataTables::of($locations)
        ->addIndexColumn()
        ->addColumn('location', fn($row) => '<strong>' . $row->name . '</strong>')
        ->addColumn('type', fn($row) => '<span class="text-muted">' . $row->type . '</span>')
        ->addColumn('total_items', fn($row) => number_format($row->total_items))
    //    ->addColumn('in_stock', fn($row) => '<span class="badge bg-success-subtle text-success">Available</span>')
      ->addColumn('in_stock', fn($row) =>
    '<span class="badge bg-success-subtle text-success">'
    . number_format($row->in_stock) .
    '</span>'
)

    ->addColumn('out_of_stock', fn($row) => '<span class="badge bg-danger-subtle text-danger">' . $row->out_of_stock . '</span>')
        ->addColumn('total_stock_qty', fn($row) => number_format($row->total_stock_qty ?? 0))
        ->addColumn('action', fn($row) => '
            <a href="' . route('stock.view', $row->id) . '" class="btn btn-sm btn-light">
                <i class="ti ti-eye"></i>
            </a>
            <a href="' . route('stock.edit', $row->id) . '" class="btn btn-sm btn-warning ms-1">
                <i class="ti ti-edit"></i>
            </a>
        ')
        ->rawColumns(['type', 'location', 'in_stock', 'out_of_stock', 'action'])
        ->make(true);
}
     public function viewStocks(Location $location, Request $request)
    {
        if ($request->ajax()) {

            $stocks = Stock::with(['productVariant.product'])
                ->where('location_id', $location->id);

            return DataTables::of($stocks)
                ->addIndexColumn()

                ->addColumn('product', function ($row) {
                    return optional($row->productVariant->product)->name ?? '-';
                })

                ->addColumn('variant', function ($row) {
    $variant = $row->productVariant;

    if (!$variant) {
        return '-';
    }

    // Build a nice HTML card for the variant
    $html = '<div class="variant-card p-2 border rounded bg-light">';
    $html .= '<strong>SKU:</strong> ' . $variant->sku . '<br>';
    $html .= '<strong>Batch No:</strong> ' . $variant->batch_no . '<br>';
    $html .= '<strong>MRP:</strong> ₹' . number_format($variant->mrp, 2) . '<br>';
    $html .= '<strong>Offer Price:</strong> <span class="text-success">₹' . number_format($variant->offer_price, 2) . '</span><br>';
    $html .= '<strong>Unit:</strong> ' . $variant->unit . '<br>';
    $html .= '<strong>Dimensions:</strong> ' . $variant->dimensions . '<br>';
    $html .= '<strong>Color:</strong> <span style="display:inline-block;width:15px;height:15px;background:' . $variant->color_code . ';border:1px solid #ccc;margin-right:5px;"></span>' . $variant->color_name . '<br>';
    $html .= '<strong>Material:</strong> ' . $variant->material_or_fabric_composition . '<br>';
    $html .= '<strong>GSM:</strong> ' . $variant->gsm . '<br>';
    $html .= '<strong>Weight:</strong> ' . $variant->weight . ' kg<br>';
    $html .= '<strong>Pattern:</strong> ' . $variant->pattern . '<br>';
    $html .= '<strong>MOQ:</strong> ' . $variant->moq . '<br>';
    $html .= '</div>';

    return $html;
})



                ->addColumn('available_qty', function ($row) {
                    return $row->available_qty;
                })

                ->addColumn('reorder_level', function ($row) {
                    return $row->reorder_level ?? '-';
                })

                ->addColumn('status', function ($row) {
                    if ($row->available_qty <= 0) {
                        return '<span class="badge bg-danger-subtle text-danger">Out of Stock</span>';
                    }

                    if ($row->reorder_level && $row->available_qty <= $row->reorder_level) {
                        return '<span class="badge bg-warning-subtle text-warning">Low Stock</span>';
                    }

                    return '<span class="badge bg-success-subtle text-success">Available</span>';
                })

                ->rawColumns(['status','variant'])
                ->make(true);
        }

        return view('admin.stock.view', compact('location'));
    }

    public function view($id, Request $request)
{
    $location = Location::findOrFail($id);

    if ($request->ajax()) {
        $stocks = Stock::with(['productVariant.product'])
            ->where('location_id', $id);

        return DataTables::of($stocks)
            ->addIndexColumn()
            ->make(true);
    }

    return view('admin.stock.view', compact('location'));
}

 public function edit($id, Request $request)
{
    return view('admin.stock.edit');
}

//************location  ******************/


public function savelocations(Request $request)
{
    /* -----------------------------
     | Validation (Mandatory fields)
     |-----------------------------*/
    $request->validate([
        'type' => 'required',
        'name' => 'required|string|max:255',
        'display_name' => 'required|string|max:255',

        'city' => 'required|string|max:255',
        'state' => 'required|string|max:255',
        'country' => 'required|string|max:255',

        'contact_person_name' => 'required|string|max:255',
        'phone' => 'required|string|max:20',

        'status' => 'required|in:Active,Temporarily Closed,Inactive',
    ]);

    /* -----------------------------
     | Auto-generate Location Code
     |-----------------------------*/
    $nextId = (Location::max('id') ?? 0) + 1;

    $locationCode = 'LOC-' .
        strtoupper(substr($request->type, 0, 2)) . '-' .
        str_pad($nextId, 4, '0', STR_PAD_LEFT);

    /* -----------------------------
     | Create Location
     |-----------------------------*/
    Location::create([
        'location_code' => $locationCode,

        'type' => $request->type,
        'name' => ucfirst($request->name),
        'display_name' => $request->display_name,

        // Address
        'city' => $request->city,
        'district' => $request->district,
        'state' => $request->state,
        'country' => $request->country,
        'pincode' => $request->pincode,

        // Contact
        'contact_person_name' => $request->contact_person_name,
        'phone' => $request->phone,
        'alternate_phone' => $request->alternate_phone,
        'email' => $request->email,

        // Legal & Billing
        'gst_number' => $request->gst_number,
        'pan' => $request->pan,
        'trade_license_no' => $request->trade_license_no,
        'billing_name' => $request->billing_name,
        'billing_address' => $request->billing_address,

        // Reporting
        'reporting_location_id' => $request->reporting_location_id,

        // Status & Audit
        'status' => $request->status,
        'created_by' => Auth::id(),
    ]);

    return response()->json([
        'success' => 'Location added successfully!'
    ]);
}


   public function getlocations()
{
    $locations = Location::with('reportingLocation')
        ->orderBy('id', 'DESC');

    return DataTables::of($locations)
        ->addIndexColumn()

        ->addColumn('location_code', fn ($row) => $row->location_code)
        ->addColumn('name', fn ($row) => $row->name)
        ->addColumn('type', fn ($row) => $row->type)
        ->addColumn('city', fn ($row) => $row->city ?? '-')
        ->addColumn('phone', fn ($row) => $row->phone ?? '-')
        ->addColumn('gst_number', fn ($row) => $row->gst_number ?? '-')
        ->addColumn('trade_license_no', fn ($row) => $row->trade_license_no ?? '-')

        ->addColumn('reporting_location', function ($row) {
            return $row->reportingLocation
                ? $row->reportingLocation->name
                : '-';
        })

        ->addColumn('status', function ($row) {
            $badgeClass = match ($row->status) {
                'Active' => 'success',
                'Temporarily Closed' => 'warning',
                'Inactive' => 'danger',
                default => 'secondary'
            };

            return '<span class="badge bg-' . $badgeClass . '">' . $row->status . '</span>';
        })

        ->addColumn('action', function ($row) {
            return '
            <a class="btn btn-sm btn-primary"
               href="' . route('edit-location', ['id' => $row->id]) . '">
                Edit
            </a>
            <button class="btn btn-sm btn-danger deletelocation"
                data-id="' . $row->id . '">
                Delete
            </button>';
        })

        ->rawColumns(['status', 'action'])
        ->make(true);
}

   public function editLocation(Request $request, $id)
{
    $location = Location::findOrFail($id);
      return view('admin.stock.edit_location',compact('location'));
}
   public function update(Request $request)
{
    $location = Location::findOrFail($request->id);

    // ----------------------------- 
    // Validation for mandatory fields
    // -----------------------------
    $request->validate([
        'type' => 'required',
        'name' => 'required|string|max:255',
        'display_name' => 'required|string|max:255',
        'city' => 'required|string|max:255',
        'state' => 'required|string|max:255',
        'country' => 'required|string|max:255',
        'contact_person_name' => 'required|string|max:255',
        'phone' => 'required|string|max:20',
        'status' => 'required|in:Active,Temporarily Closed,Inactive',
    ]);

    // ----------------------------- 
    // Update the location
    // -----------------------------
    $location->update([
        'type' => $request->type,
        'name' => ucfirst($request->name),
        'display_name' => $request->display_name,

        // Address
        'city' => $request->city,
        'district' => $request->district,
        'state' => $request->state,
        'country' => $request->country,
        'pincode' => $request->pincode,

        // Contact
        'contact_person_name' => $request->contact_person_name,
        'phone' => $request->phone,
        'alternate_phone' => $request->alternate_phone,
        'email' => $request->email,

        // Legal & Billing
        'gst_number' => $request->gst_number,
        'pan' => $request->pan,
        'trade_license_no' => $request->trade_license_no,
        'billing_name' => $request->billing_name,
        'billing_address' => $request->billing_address,

        // Reporting
        'reporting_location_id' => $request->reporting_location_id,

        // Status & Audit
        'status' => $request->status,
        'updated_by' => Auth::id(),
    ]);

    return response()->json([
        'success' => 'Location updated successfully!'
    ]);
}


    public function destroy($id)
    {
        $subcat = Location::findOrFail($id);
        $subcat->delete();
        return response()->json(['success' => 'Sub location deleted successfully.']);
    }
}
